# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

Python-based Raspberry Pi card scanner for MTG Bucket Catalogue. Captures and uploads card images using VL6180X distance sensor detection, Picamera2, OLED screen feedback, and buzzer audio cues. Communicates with backend via WebSocket for scan coordination and HTTP for image uploads.

## Dependencies

Install system packages and Python dependencies:

```bash
sudo apt-get install -y python3-pip i2c-tools
pip3 install adafruit-circuitpython-ssd1306 adafruit-circuitpython-vl6180x pillow aiohttp aiofiles picamera2 RPi.GPIO opencv-python numpy gpiod
```

## Running the Scanner

```bash
python3 scanner.py
```

The scanner runs on port 8000 and displays its IP address on the OLED screen.

## Linting

```bash
# From parent directory
ruff check rpi-scanner-code-v2/
ruff check --fix rpi-scanner-code-v2/
```

## Architecture

### Main Entry Point
- **scanner.py**: Initializes GPIO, camera, screen, buzzer, motion handler, WebSocket client, and local web server. Runs two async tasks concurrently.

### Core Components

**MotionHandler** (`lib/motion_handler.py`):
- Manages Picamera2 camera capture with sharpness-based auto-stabilization
- VL6180X Time-of-Flight distance sensor coordination via CardDetector
- Dual mode operation: test mode (saves to disk) and production mode (uploads to server)
- Configurable sharpness detection parameters for optimal image quality
- Handles scan count tracking and bucket completion notifications
- Pause/resume functionality with state management
- Camera configuration with rotatable output

**CardDetector** (`lib/card_detector.py`):
- VL6180X-based card detection with hardware interrupts and software debouncing
- Uses "level low" interrupt mode: fires when distance < threshold
- Non-blocking asyncio implementation with gpiod for GPIO handling
- Hardware reset capability via SHUT pin for sensor recovery
- Configurable detection threshold, measurement period, and debounce windows
- Independent sensor management that can be reset as needed

**WebSocketClient** (`lib/websocket_client.py`):
- Maintains persistent WebSocket connection to backend at `/api/scanner/ws`
- Auto-reconnects with configurable retry limits
- Message types: `ping`, `prepare` (start scan session), `stop`, `pause`, `resume`
- Loads configuration from `config.json` (server URL and access token)
- Skips connection attempts when handler is in test mode
- Dynamic protocol detection (ws:// vs wss://) based on server URL

**ScannerServer** (`lib/server.py`):
- Local aiohttp web server on port 8000
- Configuration UI at `/` for setting server URL, access token, camera rotation and resolution
- Test mode UI at `/test` for standalone testing without backend
- Live image preview at `/latest` endpoint with proper HTTP caching
- REST API for test mode control: GET/POST `/test_mode`

**Camera** (`lib/camera.py`):
- Picamera2 abstraction with configurable resolution presets
- Support for rotation configuration via config
- Clean separation of camera hardware from motion handling logic
- Resolution presets: maximum (2592x1944), high (1920x1440), medium (1296x972), low (640x480)

**AutoSizeOLED** (`lib/screen.py`):
- I2C SSD1306 OLED display (128x64) with auto-sizing text rendering
- Async queue-based updates to prevent blocking
- Methods: `show_text()`, `show_check()`, `show_cross()`, `show_wait()`
- Supports vertical flip for mounting orientation
- Background task for non-blocking display updates

**Buzzer** (`lib/buzzer.py`):
- PWM-based audio feedback using GPIO pin 26
- Fire-and-forget async methods with locking to prevent overlapping sounds
- Sounds: `ready()`, `error()`, `wait()`, `quick_beep()`, `friendly_beep()`

**Config** (`lib/config.py`):
- Centralized configuration management with defaults
- Camera rotation and resolution configuration support
- JSON-based storage with graceful fallbacks

### Hardware Configuration

- VL6180X ToF distance sensor: I2C address 0x29, interrupt pin GPIO 4, SHUT pin GPIO 27
- Buzzer: GPIO pin 26
- OLED screen: I2C address 0x3C (SCL pin 5, SDA pin 3)
- Camera: Picamera2 with configurable resolutions (default maximum: 2592x1944)
- Shared I2C bus with locking for thread safety

### Configuration

**config.json** (created via web UI):
```json
{
  "server_url": "http://192.168.0.192:8080",
  "access_token": "your_token_here",
  "camera_rotation": 180,
  "camera_resolution": "maximum"
}
```

Camera resolution options: "maximum", "high", "medium", "low"

### Test Mode

Test mode allows standalone operation without backend connection:
- Access UI at `http://{scanner_ip}:8000/test`
- Saves captured images to `./out/latest.jpg` with atomic file operations
- Disables WebSocket connection attempts automatically
- View live preview at `/latest` endpoint with proper caching headers
- Toggle via REST API: `POST /test_mode`

### Scanning Flow

1. WebSocket receives `prepare` message with bucket info or test mode enabled
2. CardDetector initializes VL6180X sensor and starts continuous ranging
3. When distance drops below 25mm (card enters), sensor triggers interrupt
4. Software debouncing waits 200ms to stabilize sensor readings
5. MotionHandler processes callback and starts sharpness-optimized capture process
6. Wait for stable image using Laplacian variance analysis in separate task
7. Image processed with configured rotation and uploaded/saved accordingly
8. Screen shows progress, buzzer provides audio feedback
9. On bucket completion, sends notification to `/api/scanner/bucket_scan_finished`

### VL6180X Distance Sensor

Time-of-Flight sensor configuration:
- Entry threshold: 25mm (card detection)
- Debounce window: 200ms for stable detection
- Distance sensor runs autonomously with 5ms measurement intervals
- Hardware interrupt generation when threshold crossed
- Software debouncing prevents false triggers from sensor noise
- Hardware reset capability via SHUT pin for sensor recovery
- Uses gpiod library for modern GPIO handling

### Sharpness Detection

Optimizes capture timing using Laplacian variance analysis:
- Configurable parameters: max wait time, stability thresholds, resize factor
- Early exit when "good enough" sharpness reached (150.0 default)
- Dynamic sampling rate adjusts for processing overhead
- Performance tracking and detailed timing measurements
- Falls back to fixed wait time if detection fails
- Supports configurable minimum stabilization time

## Development Notes

- All async operations use asyncio event loop from `scanner.py`
- VL6180X uses gpiod for modern GPIO interrupt handling
- Single I2C bus with locking for thread safety between components
- Display updates are non-blocking via dedicated queue task
- Buzzer operations use fire-and-forget decorator with async locks
- Atomic file operations for test mode image saves
- Image uploads include `collection_id` and `bucket_label` in form data
- Test scripts in `test-scripts/` for isolated component testing
- Newer sensor tests in `newer-sensors-tests/` for VL6180X development
- Hardware configuration constants in `lib/motion_handler.py` and `lib/card_detector.py`
- This code is synced via rsync to a rpi and executed by the user. This current code is not in the rpi
