import RPi.GPIO as GPIO
import asyncio
import functools


def fire_and_forget(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        coro = func(*args, **kwargs)
        asyncio.create_task(coro)

    return wrapper


class Buzzer:
    def __init__(self, pin=26):
        self._lock = asyncio.Lock()
        self.pin = pin
        GPIO.setwarnings(False)
        GPIO.setmode(GPIO.BCM)
        GPIO.setup(self.pin, GPIO.OUT)
        self.pwm = GPIO.PWM(self.pin, 440)

    def cleanup(self):
        self.pwm.stop()
        GPIO.cleanup()

    @fire_and_forget
    async def ready(self, repeats=1):
        async with self._lock:
            self.pwm.start(50)
            for _ in range(repeats):
                for freq in [392, 440, 494, 523]:
                    self.pwm.ChangeFrequency(freq)
                    await asyncio.sleep(0.15)
            self.pwm.stop()

    @fire_and_forget
    async def error(self, repeats=2):
        async with self._lock:
            self.pwm.start(50)
            for _ in range(repeats):
                self.pwm.ChangeFrequency(440)
                await asyncio.sleep(0.15)
                self.pwm.ChangeFrequency(494)
                await asyncio.sleep(0.05)
                self.pwm.ChangeFrequency(440)
                await asyncio.sleep(0.2)
            self.pwm.stop()

    @fire_and_forget
    async def wait(self, repeats=1):
        async with self._lock:
            self.pwm.start(50)
            for _ in range(repeats):
                for freq in [523, 494, 440, 494]:
                    self.pwm.ChangeFrequency(freq)
                    await asyncio.sleep(0.2)
            self.pwm.stop()

    @fire_and_forget
    async def friendly_beep(self):
        async with self._lock:
            self.pwm.start(50)
            self.pwm.ChangeFrequency(587)
            await asyncio.sleep(0.06)
            self.pwm.ChangeFrequency(740)
            await asyncio.sleep(0.06)
            self.pwm.stop()

    @fire_and_forget
    async def quick_beep(self):
        async with self._lock:
            self.pwm.start(50)
            self.pwm.ChangeFrequency(65.41)
            await asyncio.sleep(0.07)
            self.pwm.stop()
