#!/usr/bin/env python3
"""
VL6180X card detector using hardware interrupts with software debouncing.

- Uses "level low" interrupt mode: fires when distance < threshold
- Debounces interrupts: only counts a card after configured silence period
- Non-blocking asyncio implementation with gpiod for interrupt handling
"""

import asyncio
import time
from typing import Optional, Callable, Awaitable

import adafruit_vl6180x
import gpiod
from gpiod.line import Bias, Edge, Direction, Value


SYSTEM__MODE_GPIO1 = 0x011
SYSTEM__INTERRUPT_CONFIG_GPIO = 0x014
SYSTEM__INTERRUPT_CLEAR = 0x015
SYSRANGE__START = 0x018
SYSRANGE__THRESH_LOW = 0x01A
SYSRANGE__INTERMEASUREMENT_PERIOD = 0x01B
SYSRANGE__MAX_CONVERGENCE_TIME = 0x01C
RESULT__INTERRUPT_STATUS_GPIO = 0x04F
RESULT__RANGE_VAL = 0x062

INT_STATUS_MASK = 0x07
INT_SRC_LEVEL_LOW = 0x01


class CardDetector:
    """VL6180X-based card detector with debounced interrupt handling."""

    def __init__(
        self,
        i2c_bus,
        on_card_detected: Callable[[], Awaitable[None]],
        i2c_lock: asyncio.Lock = None,
        int_pin: int = 4,
        shut_pin: int = 27,
        threshold_mm: int = 50,
        loop_period_ms: int = 5,
        debounce_s: float = 0.2,
    ) -> None:
        self._i2c_bus = i2c_bus
        self._i2c_lock = i2c_lock if i2c_lock is not None else asyncio.Lock()
        self._on_card_detected = on_card_detected
        self._int_pin = int_pin
        self._shut_pin = shut_pin
        self._threshold_mm = threshold_mm
        self._loop_period_ms = loop_period_ms
        self._debounce_window_s = debounce_s

        print("[CardDetector] Initializing VL6180X sensor on I2C bus")
        self._sensor = adafruit_vl6180x.VL6180X(i2c_bus)

        print(
            f"[CardDetector] Sensor object created, using GPIO pin {int_pin} for interrupts, debounce window: {debounce_s}s"
        )

        self._last_distance: Optional[int] = None
        self._debounce_task: Optional[asyncio.Task] = None
        self._interrupt_task: Optional[asyncio.Task] = None
        self._request = None
        self._loop = None

    @staticmethod
    async def _hardware_reset_via_shut_pin(shut_pin: int) -> None:
        """Perform hardware reset by toggling SHUT pin."""
        print(f"[CardDetector] Performing hardware reset via SHUT pin {shut_pin}")

        request = gpiod.request_lines(
            "/dev/gpiochip0",
            consumer="vl6180x-reset",
            config={
                shut_pin: gpiod.LineSettings(
                    direction=Direction.OUTPUT,
                    output_value=Value.ACTIVE,
                )
            },
        )

        try:
            request.set_value(shut_pin, Value.INACTIVE)
            print("[CardDetector] SHUT pin pulled low, sensor powered off")
            await asyncio.sleep(0.05)

            request.set_value(shut_pin, Value.ACTIVE)
            print("[CardDetector] SHUT pin pulled high, sensor powered on")
            await asyncio.sleep(0.1)

        finally:
            request.release()

        print("[CardDetector] Hardware reset complete")

    async def _configure_sensor_registers(self) -> None:
        """Configure VL6180X sensor registers."""
        async with self._i2c_lock:
            print("[CardDetector] Configuring sensor registers:")
            print("  - GPIO1 mode: 0x10 (interrupt output)")
            self._sensor._write_8(SYSTEM__MODE_GPIO1, 0x10)

            print("  - Interrupt config: 0x01 (level low mode)")
            self._sensor._write_8(SYSTEM__INTERRUPT_CONFIG_GPIO, 0x01)

            print(f"  - Low threshold: {self._threshold_mm}mm")
            self._sensor._write_8(SYSRANGE__THRESH_LOW, self._threshold_mm)

            print(f"  - Measurement period: {self._loop_period_ms}ms")
            self._sensor._write_8(SYSRANGE__INTERMEASUREMENT_PERIOD, self._loop_period_ms)

            print("  - Max convergence time: 0x0A")
            self._sensor._write_8(SYSRANGE__MAX_CONVERGENCE_TIME, 0x0A)

            print("  - Clearing interrupts: 0x07")
            self._sensor._write_8(SYSTEM__INTERRUPT_CLEAR, 0x07)

            print("  - Starting continuous ranging: 0x03")
            self._sensor._write_8(SYSRANGE__START, 0x03)

    async def reset_sensor(self) -> None:
        """Reset the sensor and reconfigure it."""
        print("[CardDetector] Resetting sensor...")

        if not self._request:
            print("[CardDetector] GPIO not yet initialized, initializing now...")
            await self.init_gpio()

        if hasattr(self, "_sensor") and self._sensor:
            try:
                async with self._i2c_lock:
                    self._sensor._write_8(SYSRANGE__START, 0x00)
                await self.clear_interrupt()
            except Exception as e:
                print(f"[CardDetector] Warning: Could not stop sensor cleanly: {e}")

        await self._hardware_reset_via_shut_pin(self._shut_pin)

        print("[CardDetector] Recreating sensor object (full initialization with fresh-start flag)...")
        async with self._i2c_lock:
            self._sensor = adafruit_vl6180x.VL6180X(self._i2c_bus)
        print("[CardDetector] Sensor object recreated")

        await self._configure_sensor_registers()

        print("[CardDetector] Waiting 500ms for measurements to stabilize...")
        await asyncio.sleep(0.5)

        async with self._i2c_lock:
            initial_distance = self._sensor._read_8(RESULT__RANGE_VAL)
            initial_status = self._sensor._read_8(RESULT__INTERRUPT_STATUS_GPIO) & INT_STATUS_MASK
        print("[CardDetector] Post-reset sensor readings:")
        print(f"  - Distance: {initial_distance}mm")
        print(f"  - Interrupt status: 0x{initial_status:02X}")
        print("[CardDetector] Sensor reset complete")

    async def read_distance(self) -> int:
        async with self._i2c_lock:
            distance = self._sensor._read_8(RESULT__RANGE_VAL)
            self._last_distance = distance
            return distance

    async def clear_interrupt(self) -> None:
        async with self._i2c_lock:
            self._sensor._write_8(SYSTEM__INTERRUPT_CLEAR, 0x07)

    async def interrupt_status(self) -> int:
        async with self._i2c_lock:
            return self._sensor._read_8(RESULT__INTERRUPT_STATUS_GPIO) & INT_STATUS_MASK

    async def _debounced_card_detected(self) -> None:
        print(f"[CardDetector] ⏱️  Debounce timer started ({self._debounce_window_s}s)")
        await asyncio.sleep(self._debounce_window_s)
        try:
            current_distance = await self.read_distance()
            print(f"[CardDetector] 🃏 CARD DETECTED! Debounce complete, current distance: {current_distance}mm")
            print("[CardDetector] Calling card detected callback...")
            await self._on_card_detected()
            print("[CardDetector] Card detected callback completed")
        except OSError as e:
            print(f"[CardDetector] ⚠️  I2C error during debounce handling: {e}")

    async def _handle_interrupt_async(self) -> None:
        timestamp = time.time()
        print(f"[CardDetector] ⚡ INTERRUPT at [{timestamp:.3f}]")

        try:
            status = await self.interrupt_status()
        except OSError as e:
            print(f"[CardDetector] ⚠️  I2C error reading interrupt status: {e}")
            try:
                await self.clear_interrupt()
            except OSError as clear_error:
                print(f"[CardDetector] ⚠️  I2C error clearing interrupt: {clear_error}")
            return

        if status != INT_SRC_LEVEL_LOW:
            print(f"[CardDetector] ⚠️  Status mismatch - 0x{status:02X}, not level low interrupt, clearing and ignoring")
            try:
                await self.clear_interrupt()
            except OSError as e:
                print(f"[CardDetector] ⚠️  I2C error clearing interrupt: {e}")
            return

        print("[CardDetector] ✓ Valid level low interrupt detected")

        if self._debounce_task and not self._debounce_task.done():
            print("[CardDetector] ⏱️  Cancelling previous debounce task (retriggered)")
            self._debounce_task.cancel()

        print("[CardDetector] Starting new debounce task")
        self._debounce_task = asyncio.create_task(self._debounced_card_detected())

        try:
            await self.clear_interrupt()
            print("[CardDetector] Interrupt cleared")
        except OSError as e:
            print(f"[CardDetector] ⚠️  I2C error clearing interrupt: {e}")

    def handle_interrupt(self) -> None:
        if self._interrupt_task and not self._interrupt_task.done():
            print("[CardDetector] Previous interrupt task still running, skipping")
            return
        self._interrupt_task = asyncio.create_task(self._handle_interrupt_async())

    async def init_gpio(self) -> None:
        """Initialize GPIO interrupt handling only (deferred sensor init)."""
        print("[CardDetector] Initializing GPIO interrupt handling...")

        self._loop = asyncio.get_running_loop()

        print("[CardDetector] Configuring GPIO interrupt:")
        print(f"  - GPIO pin: {self._int_pin}")
        print("  - Edge detection: FALLING")
        print("  - Bias: PULL_UP")

        self._request = gpiod.request_lines(
            "/dev/gpiochip0",
            consumer="vl6180x-detector",
            config={
                self._int_pin: gpiod.LineSettings(
                    edge_detection=Edge.FALLING,
                    bias=Bias.PULL_UP,
                )
            },
        )

        print("[CardDetector] GPIO configured successfully")
        print(f"  - File descriptor: {self._request.fd}")
        print(f"  - Monitoring for edge events on fd {self._request.fd}")

        event_count = [0]

        def on_edge_event():
            event_count[0] += 1
            print(f"[CardDetector] *** Edge event callback triggered (count: {event_count[0]}) ***")

            events = self._request.read_edge_events()
            print(f"[CardDetector] Read {len(events)} edge event(s)")

            for i, event in enumerate(events):
                event_type_name = "FALLING_EDGE" if event.event_type == event.Type.FALLING_EDGE else "RISING_EDGE"
                print(
                    f"[CardDetector] Event {i + 1}/{len(events)}: type={event_type_name}, line_offset={event.line_offset}"
                )

                if event.event_type == event.Type.FALLING_EDGE:
                    print("[CardDetector] Processing FALLING edge - calling handle_interrupt()")
                    self.handle_interrupt()
                    if self._last_distance is not None:
                        print(f"[CardDetector] Last measured distance: {self._last_distance}mm")
                else:
                    print("[CardDetector] Ignoring RISING edge event")

        self._loop.add_reader(self._request.fd, on_edge_event)
        print(f"[CardDetector] Event loop reader registered for fd {self._request.fd}")
        print(f"[CardDetector] ✓ GPIO initialized and listening for interrupts on GPIO {self._int_pin}")

    async def start(self) -> None:
        """Initialize sensor and register interrupt handler with event loop."""
        print("[CardDetector] Starting card detector (full initialization)...")

        await self.reset_sensor()

        if not self._loop:
            self._loop = asyncio.get_running_loop()

        if not self._request:
            await self.init_gpio()

        current_distance = await self.read_distance()
        current_status = await self.interrupt_status()
        print("[CardDetector] Pre-start sensor status:")
        print(f"  - Distance: {current_distance}mm")
        print(f"  - Interrupt status: 0x{current_status:02X}")

        print(f"[CardDetector] ✓ Card detector started and listening for interrupts on GPIO {self._int_pin}")

    def shutdown(self) -> None:
        """Stop sensor and clean up resources."""
        print("[CardDetector] Shutting down card detector...")

        if self._loop and self._request:
            print(f"[CardDetector] Removing event loop reader for fd {self._request.fd}")
            self._loop.remove_reader(self._request.fd)

        if self._request:
            print("[CardDetector] Closing GPIO line request")
            self._request.close()

        print("[CardDetector] Stopping sensor ranging")
        try:
            self._loop.run_until_complete(self._shutdown_i2c_ops())
        except RuntimeError:
            print("[CardDetector] Event loop not available, performing sync shutdown")
            self._sensor._write_8(SYSRANGE__START, 0x00)
        print("[CardDetector] ✓ Shutdown complete")

    async def _shutdown_i2c_ops(self) -> None:
        """Async wrapper for I2C shutdown operations."""
        async with self._i2c_lock:
            self._sensor._write_8(SYSRANGE__START, 0x00)
        await self.clear_interrupt()
