import asyncio
import adafruit_ssd1306
import board
import busio
from PIL import Image, ImageDraw, ImageFont

# GND      ->  GND (Pin 6)
# VCC      ->  3.3V (Pin 1)
# SCL      ->  SCL (Pin 5)
# SDA      ->  SDA (Pin 3)


class AutoSizeOLED:
    def __init__(self, i2c_bus=None, i2c_lock=None, i2c_addr=0x3C, max_lines=3, flip_vertical=False):
        if i2c_bus is None:
            i2c_bus = busio.I2C(board.SCL, board.SDA)
        self.display = adafruit_ssd1306.SSD1306_I2C(128, 64, i2c_bus, addr=i2c_addr)
        self._i2c_lock = i2c_lock if i2c_lock is not None else asyncio.Lock()
        self.font_path = "/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf"
        self.max_lines = max_lines
        self.flip_vertical = flip_vertical
        self._queue = asyncio.Queue()
        self._task = asyncio.create_task(self._display_loop())

    async def _display_loop(self):
        """Background task to process display updates from the queue."""
        while True:
            image = await self._queue.get()
            await self._display_image(image)
            self._queue.task_done()

    async def _display_image(self, image):
        """Helper method to handle common display logic."""
        async with self._i2c_lock:
            self.display.fill(0)
            if self.flip_vertical:
                image = image.transpose(Image.ROTATE_180)
            self.display.image(image)
            self.display.show()

    def show_text(self, text):
        """Fire and forget method to display text."""
        image = self._create_text_image(text)
        self._queue.put_nowait(image)

    def show_check(self, message=None):
        """Fire and forget method to display a checkmark."""
        image = self._create_check_image(message)
        self._queue.put_nowait(image)

    def show_cross(self, message=None):
        """Fire and forget method to display a cross."""
        image = self._create_cross_image(message)
        self._queue.put_nowait(image)

    def show_wait(self, message="Wait"):
        """Fire and forget method to display a wait message."""
        image = self._create_text_image(message)
        self._queue.put_nowait(image)

    def _create_text_image(self, text):
        """Helper method to create an image with text."""
        image = Image.new("1", (self.display.width, self.display.height))
        draw = ImageDraw.Draw(image)
        for size in range(40, 8, -2):
            font = ImageFont.truetype(self.font_path, size)
            wrapped = []
            words = text.split()

            # First check if any individual word is too wide for this font size
            too_wide = False
            for word in words:
                bbox = draw.textbbox((0, 0), word, font=font)
                if bbox[2] - bbox[0] > self.display.width:
                    too_wide = True
                    break
            if too_wide:
                continue

            current_line = words[0]
            for word in words[1:]:
                test_line = current_line + " " + word
                bbox = draw.textbbox((0, 0), test_line, font=font)
                width = bbox[2] - bbox[0]
                if width <= self.display.width:
                    current_line = test_line
                else:
                    wrapped.append(current_line)
                    current_line = word
            wrapped.append(current_line)

            if len(wrapped) > self.max_lines:
                continue

            # Get line height using font metrics
            bbox = draw.textbbox((0, 0), "hg", font=font)
            line_height = bbox[3] - bbox[1]
            total_height = line_height * len(wrapped)

            if total_height <= self.display.height:
                y = (self.display.height - total_height) // 2
                for line in wrapped:
                    bbox = draw.textbbox((0, 0), line, font=font)
                    text_width = bbox[2] - bbox[0]
                    x = (self.display.width - text_width) // 2
                    draw.text((x, y), line, font=font, fill=255)
                    y += line_height
                break
        return image

    def _create_check_image(self, message=None):
        """Helper method to create an image with a checkmark."""
        image = Image.new("1", (self.display.width, self.display.height))
        draw = ImageDraw.Draw(image)
        base_y = 16 if message else 32
        text_area_height = 20 if message else 0
        size = 40
        offset = (self.display.width - size) // 2
        draw.line(
            (offset, base_y - size // 4, offset + size // 3, base_y + size // 4),
            fill=255,
            width=3,
        )
        draw.line(
            (offset + size // 3, base_y + size // 4, offset + size, base_y - size // 2),
            fill=255,
            width=3,
        )
        if message:
            self._draw_bottom_text(draw, message, text_area_height)
        return image

    def _create_cross_image(self, message=None):
        """Helper method to create an image with a cross."""
        image = Image.new("1", (self.display.width, self.display.height))
        draw = ImageDraw.Draw(image)
        base_y = 12 if message else 32
        text_area_height = 30 if message else 0
        size = 24
        offset = (self.display.width - size) // 2
        draw.line(
            (offset, base_y - size // 2, offset + size, base_y + size // 2),
            fill=255,
            width=3,
        )
        draw.line(
            (offset + size, base_y - size // 2, offset, base_y + size // 2),
            fill=255,
            width=3,
        )
        if message:
            self._draw_bottom_text(draw, message, text_area_height)
        return image

    def _draw_bottom_text(self, draw, message, text_area_height):
        """Helper to draw text at bottom with automatic sizing."""
        max_font_size = 20
        for size in range(max_font_size, 8, -1):
            font = ImageFont.truetype(self.font_path, size)
            bbox = draw.textbbox((0, 0), message, font=font)
            text_width = bbox[2] - bbox[0]
            text_height = bbox[3] - bbox[1]
            if text_width <= self.display.width and text_height <= text_area_height:
                x = (self.display.width - text_width) // 2
                y = (
                    self.display.height
                    - text_area_height
                    + (text_area_height - text_height) // 2
                )
                draw.text((x, y), message, font=font, fill=255)
                break

    async def close(self):
        """Cleanup resources."""
        self._task.cancel()
        try:
            await self._task
        except asyncio.CancelledError:
            pass
