from aiohttp import web
from pathlib import Path
import asyncio
from email.utils import formatdate
from lib.config import load_config, save_config


class ScannerServer:
    def __init__(self, handler, screen, buzzer, host="0.0.0.0", port=8000, ws_client=None):
        self.host = host
        self.port = port
        self.handler = handler
        self.buzzer = buzzer
        self.screen = screen
        self.app = web.Application()
        self._setup_routes()
        self.runner = None
        self.ws_client = ws_client

    def _setup_routes(self):
        self.app.router.add_get("/", self.handle_index)
        self.app.router.add_post("/", self.handle_config_post)
        self.app.router.add_get("/latest", self.handle_latest)
        self.app.router.add_get("/test_mode", self.handle_test_mode_get)
        self.app.router.add_post("/test_mode", self.handle_test_mode_post)
        self.app.router.add_get("/test", self.handle_test_page)

    async def handle_index(self, request):
        config = load_config()
        with open(Path(__file__).parent.parent / "templates" / "config.html", "r") as f:
            content = f.read()
            content = content.replace("{{ server_url }}", config["server_url"])
            content = content.replace(
                "{{ '[TOKEN SET]' if has_token else '' }}",
                "[TOKEN SET]" if config.get("access_token") else "",
            )
            content = content.replace(
                "{{ 'Token set, not showing for security' if has_token else 'Enter new token' }}",
                ("Token set, not showing for security" if config.get("access_token") else "Enter new token"),
            )
            content = content.replace("{{ camera_rotation }}", str(config.get("camera_rotation", 180)))
            content = content.replace("{{ camera_resolution }}", config.get("camera_resolution", "maximum"))
        return web.Response(text=content, content_type="text/html")

    async def handle_config_post(self, request):
        data = await request.post()
        server_url = data["server_url"]

        if not server_url.startswith("http://") and not server_url.startswith("https://"):
            return web.Response(text="Error: Server URL must start with http:// or https://", status=400)

        camera_rotation = data.get("camera_rotation", 180)
        camera_resolution = data.get("camera_resolution", "maximum")
        save_config(server_url, data["access_token"], camera_rotation, camera_resolution)

        if self.handler:
            self.handler.reload_config()

        if self.ws_client:
            asyncio.create_task(self.ws_client.reload_config())

        raise web.HTTPFound("/")

    async def handle_latest(self, request):
        try:
            image_path = Path("./out/latest.jpg")
            with open(image_path, "rb") as f:
                response = web.Response(body=f.read(), content_type="image/jpeg")
                # Add Last-Modified header for client polling
                response.headers["Last-Modified"] = formatdate(image_path.stat().st_mtime, usegmt=True)
                return response
        except FileNotFoundError:
            raise web.HTTPNotFound(text="Image not found")

    async def handle_test_mode_get(self, request):
        """Get current test mode status"""
        test_mode_enabled = getattr(self.handler, "test_mode", False)
        return web.json_response({"enabled": test_mode_enabled})

    async def handle_test_mode_post(self, request):
        """Toggle test mode on/off"""
        try:
            data = await request.json()
            enabled = data.get("enabled", False)

            if enabled:
                # Enable test mode
                await self.handler.enable_test_mode()
                # Close WebSocket connection to prevent reconnection attempts
                if self.ws_client and self.ws_client.ws:
                    await self.ws_client.ws.close()
                    self.ws_client.ws = None
                self.screen.show_text("Test Mode")
                self.buzzer.ready()
            else:
                # Disable test mode
                self.handler.disable_test_mode()
                self.screen.show_text("Ready")

            return web.json_response({"success": True, "enabled": enabled})
        except Exception as e:
            return web.json_response({"success": False, "error": str(e)}, status=400)

    async def handle_test_page(self, request):
        """Serve the dedicated test mode page"""
        with open(Path(__file__).parent.parent / "templates" / "test_mode.html", "r") as f:
            content = f.read()
        return web.Response(text=content, content_type="text/html")

    async def start(self):
        print("Starting Scanner Web Server...")
        self.runner = web.AppRunner(self.app)
        await self.runner.setup()
        site = web.TCPSite(self.runner, self.host, self.port)
        await site.start()
        print(f"Server running at http://{self.host}:{self.port}/")

    async def cleanup(self):
        if self.runner:
            await self.runner.cleanup()
