import asyncio
import aiohttp
import json
from pathlib import Path
from lib.config import load_config


class WebSocketClient:
    def __init__(self, handler, screen, buzzer, ip_text):
        self.handler = handler
        self.screen = screen
        self.buzzer = buzzer
        self.ws = None
        self.session = None
        self.reconnect_delay = 5
        self.max_failed_attempts = 3
        self.failed_attempts = 0
        self._running = False
        self.config = load_config()
        self.ip_text = ip_text
        self.server = None
        self._reload_lock = asyncio.Lock()

    async def handle_message(self, msg):
        try:
            data = json.loads(msg)
            msg_type = data.get("type")
            payload = data.get("payload", {})

            if msg_type == "ping":
                if self.screen:
                    self.screen.show_text("hi there")
                if self.buzzer:
                    self.buzzer.friendly_beep()
                print("Got ping")

            elif msg_type == "prepare":
                bucket_size = payload.get("bucket_size")
                bucket_label = payload.get("bucket_label")
                collection_id = payload.get("collection_id")

                if not all([bucket_size, bucket_label, collection_id]):
                    print("Missing required fields in prepare message")
                    return

                # Get base URL from the WebSocket URL for HTTP uploads
                ws_url = self.config["server_url"]
                base_url = ws_url.replace("ws://", "http://").replace("wss://", "https://")
                if base_url.endswith("/api/scanner/ws"):
                    base_url = base_url[:-3]  # Remove '/ws' but keep /api/scanner
                elif not base_url.endswith("/api/scanner"):
                    base_url = base_url.rstrip("/") + "/api/scanner"
                upload_url = f"{base_url}/upload"

                if self.handler:
                    await self.handler.set_ready(
                        upload_url,
                        bucket_size,
                        bucket_label,
                        collection_id,
                        self.config["access_token"],
                    )

            elif msg_type == "stop":
                if self.handler:
                    self.handler.reset()
                if self.screen:
                    self.screen.show_cross("Cancelled")
                if self.buzzer:
                    self.buzzer.error()

            elif msg_type == "pause":
                if self.handler:
                    self.handler.pause()
                if self.buzzer:
                    self.buzzer.quick_beep()
                print("Scanner paused via WebSocket")

            elif msg_type == "resume":
                if self.handler:
                    self.handler.resume()
                if self.buzzer:
                    self.buzzer.quick_beep()
                print("Scanner resumed via WebSocket")
            elif msg_type == "instant_capture":
                print(f"Received instant_capture message: {data}")

                if self.handler:
                    try:
                        print("Handler: Attempting instant capture...")

                        # Capture the instant image
                        success = await self.handler.capture_instant_image()

                        if success:
                            print("Handler: Instant capture successful")

                            # Upload the captured image
                            await self.upload_instant_capture()

                            if self.screen:
                                self.screen.show_check("Uploaded!")
                            if self.buzzer:
                                self.buzzer.friendly_beep()
                        else:
                            if self.screen:
                                self.screen.show_cross("Failed")
                            if self.buzzer:
                                self.buzzer.error()
                            print("Instant capture failed")

                    except Exception as e:
                        print(f"Handler: Instant capture exception: {e}")
                        if self.screen:
                            self.screen.show_cross("Error")
                        if self.buzzer:
                            self.buzzer.error()

        except json.JSONDecodeError:
            print(f"Failed to parse message: {msg}")
        except Exception as e:
            print(f"Error handling message: {e}")

    async def connect(self):
        self.config = load_config()
        if not self.config["server_url"] or not self.config["access_token"]:
            print("Missing server URL or access token in config")
            if self.screen:
                self.screen.show_text("No config\n" + self.ip_text)
            return False

        try:
            # Close existing session if it exists
            if self.session:
                await self.session.close()

            self.session = aiohttp.ClientSession()
            headers = {"Authorization": f"Bearer {self.config['access_token']}"}

            # Ensure proper WebSocket URL format
            ws_url = self.config["server_url"]

            # Determine if secure connection based on original protocol
            use_secure = ws_url.startswith("https://")

            # Extract host and port from URL
            ws_url = ws_url.replace("http://", "").replace("https://", "")
            if not ws_url.endswith("/api/scanner/ws"):
                ws_url = ws_url.rstrip("/") + "/api/scanner/ws"

            # Add appropriate WebSocket protocol
            ws_url = f"wss://{ws_url}" if use_secure else f"ws://{ws_url}"

            print(f"Connecting to WebSocket URL: {ws_url}")
            self.ws = await self.session.ws_connect(
                ws_url,
                headers=headers,
                heartbeat=30,
            )
            print("Connected to WebSocket server")
            if self.screen:
                self.screen.show_text("Connected\n" + self.ip_text)
            return True
        except Exception as e:
            print(f"Failed to connect: {e}")
            if self.screen:
                self.screen.show_cross("Conn error\n" + self.ip_text)
            if self.session:
                await self.session.close()
                self.session = None
            return False

    async def start(self):
        self._running = True
        is_retry = False
        while self._running:
            if self.handler and self.handler.test_mode:
                await asyncio.sleep(self.reconnect_delay)
                continue

            if self.failed_attempts >= self.max_failed_attempts:
                print(f"Max connection attempts ({self.max_failed_attempts}) reached. Save config to retry.")
                if self.screen:
                    self.screen.show_cross("Max retry\n" + self.ip_text)
                await asyncio.sleep(self.reconnect_delay)
                continue

            try:
                if not self.ws:
                    if is_retry and self.screen:
                        self.screen.show_text("Attempt reconnect")
                    if not await self.connect():
                        is_retry = True
                        self.failed_attempts += 1
                        await asyncio.sleep(self.reconnect_delay)
                        continue

                is_retry = False
                self.failed_attempts = 0

                async for msg in self.ws:
                    if msg.type == aiohttp.WSMsgType.TEXT:
                        await self.handle_message(msg.data)
                    elif msg.type in (
                        aiohttp.WSMsgType.CLOSED,
                        aiohttp.WSMsgType.ERROR,
                    ):
                        break

            except Exception as e:
                print(f"WebSocket error: {e}")
                if self.screen:
                    self.screen.show_cross("WS error")

            if self.ws:
                await self.ws.close()
                self.ws = None

            if self.session:
                await self.session.close()
                self.session = None

            if self._running:
                is_retry = True
                await asyncio.sleep(self.reconnect_delay)

    async def stop(self):
        self._running = False
        if self.ws:
            await self.ws.close()
        if self.session:
            await self.session.close()
            self.session = None

    async def reload_config(self):
        async with self._reload_lock:
            print("Reloading config...")
            self.config = load_config()
            self.failed_attempts = 0
            if self.ws:
                await self.ws.close()
                self.ws = None
            if self.session:
                await self.session.close()
                self.session = None

    async def upload_instant_capture(self):
        """Upload the instantly captured image to the server"""
        print("Uploading instant capture...")

        # TODO use a variable?
        instant_file = Path("./out/latest_instant.jpg")

        if not instant_file.exists():
            print("Instant capture image not found")
            return False

        # Read the image file
        with open(instant_file, "rb") as f:
            image_bytes = f.read()

        # Create form data
        data = aiohttp.FormData()
        data.add_field(
            "image",
            image_bytes,
            filename="instant_capture.jpg",
            content_type="image/jpeg",
        )

        # Get the base URL from config
        ws_url = self.config["server_url"]
        base_url = ws_url.replace("ws://", "http://").replace("wss://", "https://")
        if base_url.endswith("/api/scanner/ws"):
            base_url = base_url[:-3]
        elif not base_url.endswith("/api/scanner"):
            base_url = base_url.rstrip("/") + "/api/scanner"

        upload_url = f"{base_url}/upload_instant"

        # Upload the image
        headers = {"Authorization": f"Bearer {self.config['access_token']}"}

        async with aiohttp.ClientSession() as session:
            async with session.post(upload_url, data=data, headers=headers) as response:
                if response.status != 200:
                    error_text = await response.text()
                    print(f"Upload failed: {error_text}")
                    return False
                else:
                    print("Successfully uploaded instant capture")
                    return True
