#!/usr/bin/env python3
"""
VL6180X card detector using hardware interrupts with software debouncing.

- Uses "level low" interrupt mode: fires when distance < threshold
- Debounces interrupts: only counts a card after 200ms of silence
- Non-blocking asyncio implementation with RPi.GPIO for interrupt handling
"""

import asyncio
import time
from typing import Optional

import board
import busio
import adafruit_vl6180x
import RPi.GPIO as GPIO


INT_PIN = 4
DETECTION_THRESHOLD_MM = 50
INTERMEASUREMENT_MS = 5
DEBOUNCE_WINDOW_S = 0.2

SYSTEM__MODE_GPIO1 = 0x011
SYSTEM__INTERRUPT_CONFIG_GPIO = 0x014
SYSTEM__INTERRUPT_CLEAR = 0x015
SYSRANGE__START = 0x018
SYSRANGE__THRESH_LOW = 0x01A
SYSRANGE__INTERMEASUREMENT_PERIOD = 0x01B
SYSRANGE__MAX_CONVERGENCE_TIME = 0x01C
RESULT__INTERRUPT_STATUS_GPIO = 0x04F
RESULT__RANGE_VAL = 0x062

INT_STATUS_MASK = 0x07
INT_SRC_LEVEL_LOW = 0x01


class CardDetector:
    """VL6180X-based card detector with debounced interrupt handling."""

    def __init__(self, threshold_mm: int, loop_period_ms: int, debounce_s: float) -> None:
        i2c = busio.I2C(board.SCL, board.SDA)
        self._sensor = adafruit_vl6180x.VL6180X(i2c)

        self._sensor._write_8(SYSTEM__MODE_GPIO1, 0x10)
        self._sensor._write_8(SYSTEM__INTERRUPT_CONFIG_GPIO, 0x01)
        self._sensor._write_8(SYSRANGE__THRESH_LOW, threshold_mm)
        self._sensor._write_8(SYSRANGE__INTERMEASUREMENT_PERIOD, loop_period_ms)
        self._sensor._write_8(SYSRANGE__MAX_CONVERGENCE_TIME, 0x0A)
        self._sensor._write_8(SYSTEM__INTERRUPT_CLEAR, 0x07)
        self._sensor._write_8(SYSRANGE__START, 0x03)

        self._last_distance: Optional[int] = None
        self._debounce_task: Optional[asyncio.Task] = None
        self._debounce_window_s = debounce_s

    def read_distance(self) -> int:
        distance = self._sensor._read_8(RESULT__RANGE_VAL)
        self._last_distance = distance
        return distance

    def clear_interrupt(self) -> None:
        self._sensor._write_8(SYSTEM__INTERRUPT_CLEAR, 0x07)

    def interrupt_status(self) -> int:
        return self._sensor._read_8(RESULT__INTERRUPT_STATUS_GPIO) & INT_STATUS_MASK

    async def _debounced_card_detected(self) -> None:
        await asyncio.sleep(self._debounce_window_s)
        current_distance = self.read_distance()
        print(f"🃏 CARD DETECTED! (current distance: {current_distance}mm)")

    def handle_interrupt(self, loop: asyncio.AbstractEventLoop) -> None:
        timestamp = time.time()
        status = self.interrupt_status()
        distance = self.read_distance()

        print(f"[{timestamp:.3f}] Status: 0x{status:02X}, Distance: {distance} mm")

        if status != INT_SRC_LEVEL_LOW:
            self.clear_interrupt()
            return

        if self._debounce_task and not self._debounce_task.done():
            self._debounce_task.cancel()

        self._debounce_task = asyncio.run_coroutine_threadsafe(
            self._debounced_card_detected(), loop
        )

        self.clear_interrupt()

    def shutdown(self) -> None:
        self._sensor._write_8(SYSRANGE__START, 0x00)
        self.clear_interrupt()


async def main() -> None:
    detector = CardDetector(
        threshold_mm=DETECTION_THRESHOLD_MM,
        loop_period_ms=INTERMEASUREMENT_MS,
        debounce_s=DEBOUNCE_WINDOW_S,
    )

    print(
        "VL6180X card detector (interrupt + debounce):\n"
        f"- Detection threshold: {DETECTION_THRESHOLD_MM} mm\n"
        f"- Debounce window: {DEBOUNCE_WINDOW_S * 1000:.0f} ms\n"
        f"- GPIO{INT_PIN} falling edge\n"
        "Press Ctrl+C to stop.\n"
    )

    GPIO.setmode(GPIO.BCM)
    GPIO.setwarnings(False)

    try:
        GPIO.cleanup(INT_PIN)
    except Exception:
        pass

    GPIO.setup(INT_PIN, GPIO.IN, pull_up_down=GPIO.PUD_UP)

    loop = asyncio.get_running_loop()

    def gpio_callback(channel):
        detector.handle_interrupt(loop)
        if detector._last_distance is not None:
            print(f"[DEBUG] Last distance: {detector._last_distance} mm")

    GPIO.add_event_detect(INT_PIN, GPIO.FALLING, callback=gpio_callback, bouncetime=50)

    try:
        while True:
            await asyncio.sleep(3600)
    except KeyboardInterrupt:
        pass
    finally:
        GPIO.cleanup()
        detector.shutdown()
        print("\nShutdown complete")


if __name__ == "__main__":
    asyncio.run(main())
