#!/usr/bin/env python3
"""
VL6180X card detector using hardware interrupts with software debouncing.

- Uses "level low" interrupt mode: fires when distance < threshold
- Debounces interrupts: only counts a card after 200ms of silence
- Non-blocking asyncio implementation with gpiod for interrupt handling
"""

import asyncio
import time
from typing import Optional

import board
import busio
import adafruit_vl6180x
import gpiod
from gpiod.line import Bias, Edge, Direction, Value


INT_PIN = 4
SHUT_PIN = 27
DETECTION_THRESHOLD_MM = 50
INTERMEASUREMENT_MS = 5
DEBOUNCE_WINDOW_S = 0.2

SYSTEM__MODE_GPIO1 = 0x011
SYSTEM__INTERRUPT_CONFIG_GPIO = 0x014
SYSTEM__INTERRUPT_CLEAR = 0x015
SYSRANGE__START = 0x018
SYSRANGE__THRESH_LOW = 0x01A
SYSRANGE__INTERMEASUREMENT_PERIOD = 0x01B
SYSRANGE__MAX_CONVERGENCE_TIME = 0x01C
RESULT__INTERRUPT_STATUS_GPIO = 0x04F
RESULT__RANGE_VAL = 0x062

INT_STATUS_MASK = 0x07
INT_SRC_LEVEL_LOW = 0x01


class CardDetector:
    """VL6180X-based card detector with debounced interrupt handling."""

    def __init__(self, threshold_mm: int, loop_period_ms: int, debounce_s: float) -> None:
        self._i2c = busio.I2C(board.SCL, board.SDA)
        self._sensor = adafruit_vl6180x.VL6180X(self._i2c)
        self._threshold_mm = threshold_mm
        self._loop_period_ms = loop_period_ms

        self._initialize_sensor()

        self._last_distance: Optional[int] = None
        self._debounce_task: Optional[asyncio.Task] = None
        self._debounce_window_s = debounce_s
        self._card_count = 0
        self._shut_request = None

    def _initialize_sensor(self) -> None:
        """Initialize sensor registers."""
        print("[INIT] Configuring VL6180X sensor registers...")
        self._sensor._write_8(SYSTEM__MODE_GPIO1, 0x10)
        self._sensor._write_8(SYSTEM__INTERRUPT_CONFIG_GPIO, 0x01)
        self._sensor._write_8(SYSRANGE__THRESH_LOW, self._threshold_mm)
        self._sensor._write_8(SYSRANGE__INTERMEASUREMENT_PERIOD, self._loop_period_ms)
        self._sensor._write_8(SYSRANGE__MAX_CONVERGENCE_TIME, 0x0A)
        self._sensor._write_8(SYSTEM__INTERRUPT_CLEAR, 0x07)
        self._sensor._write_8(SYSRANGE__START, 0x03)
        print("[INIT] Sensor configured and ranging started")

    def read_distance(self) -> int:
        distance = self._sensor._read_8(RESULT__RANGE_VAL)
        self._last_distance = distance
        return distance

    def clear_interrupt(self) -> None:
        self._sensor._write_8(SYSTEM__INTERRUPT_CLEAR, 0x07)

    def interrupt_status(self) -> int:
        return self._sensor._read_8(RESULT__INTERRUPT_STATUS_GPIO) & INT_STATUS_MASK

    async def _debounced_card_detected(self) -> None:
        await asyncio.sleep(self._debounce_window_s)
        current_distance = self.read_distance()
        self._card_count += 1
        print(f"🃏 CARD DETECTED! (count: {self._card_count}, current distance: {current_distance}mm)")

    def handle_interrupt(self) -> None:
        timestamp = time.time()
        status = self.interrupt_status()
        distance = self.read_distance()

        print(f"[{timestamp:.3f}] Status: 0x{status:02X}, Distance: {distance} mm")

        if status != INT_SRC_LEVEL_LOW:
            self.clear_interrupt()
            return

        if self._debounce_task and not self._debounce_task.done():
            self._debounce_task.cancel()

        self._debounce_task = asyncio.create_task(self._debounced_card_detected())

        self.clear_interrupt()

    def set_shut_request(self, shut_request) -> None:
        """Set the SHUT pin GPIO request for testing."""
        self._shut_request = shut_request

    def set_shut_pin(self, value: bool) -> None:
        """Control SHUT pin: True=active, False=shutdown."""
        if self._shut_request:
            pin_value = Value.ACTIVE if value else Value.INACTIVE
            self._shut_request.set_value(SHUT_PIN, pin_value)
            state = "HIGH (active)" if value else "LOW (shutdown)"
            print(f"[SHUT] Pin set to {state}")

    async def test_shut_functionality(self) -> None:
        """Test SHUT pin by toggling it and observing sensor behavior."""
        print("\n" + "=" * 60)
        print("[SHUT TEST] Starting SHUT pin functionality test")
        print("=" * 60)

        print("\n[SHUT TEST] Phase 1: Pulling SHUT LOW (sensor should shutdown)")
        self.set_shut_pin(False)

        await asyncio.sleep(1)

        print("[SHUT TEST] Attempting to read sensor while SHUT is LOW...")
        try:
            status = self._sensor._read_8(RESULT__INTERRUPT_STATUS_GPIO)
            distance = self._sensor._read_8(RESULT__RANGE_VAL)
            print(f"[SHUT TEST] ⚠️  UNEXPECTED: Sensor still responding! Status: 0x{status:02X}, Distance: {distance}mm")
        except Exception as e:
            print(f"[SHUT TEST] ✓ Expected I2C error during shutdown: {e}")

        print("\n[SHUT TEST] Waiting 3 seconds while SHUT is LOW...")
        print("[SHUT TEST] (Try triggering the sensor - interrupts should not fire)")
        await asyncio.sleep(3)

        print("\n[SHUT TEST] Phase 2: Pulling SHUT HIGH (sensor should resume)")
        self.set_shut_pin(True)

        print("[SHUT TEST] Waiting 100ms for sensor boot...")
        await asyncio.sleep(0.1)

        print("[SHUT TEST] Recreating sensor object (full initialization with fresh-start flag)...")
        try:
            self._sensor = adafruit_vl6180x.VL6180X(self._i2c)
            print("[SHUT TEST] ✓ Sensor object recreated (fresh start complete)")
        except Exception as e:
            print(f"[SHUT TEST] ✗ Failed to recreate sensor: {e}")
            return

        print("[SHUT TEST] Configuring sensor registers...")
        try:
            self._initialize_sensor()
            print("[SHUT TEST] ✓ Sensor configured successfully")
        except Exception as e:
            print(f"[SHUT TEST] ✗ Failed to configure: {e}")

        print("[SHUT TEST] Waiting 500ms for fresh measurements...")
        await asyncio.sleep(0.5)

        print("[SHUT TEST] Reading ambient wall distance...")
        try:
            ambient_distance = self.read_distance()
            print(f"[SHUT TEST] Ambient wall distance: {ambient_distance}mm (expected ~93mm)")
        except Exception as e:
            print(f"[SHUT TEST] ✗ Failed to read distance: {e}")

        print("\n[SHUT TEST] Test complete - normal operation should resume")
        print("=" * 60 + "\n")

    def shutdown(self) -> None:
        self._sensor._write_8(SYSRANGE__START, 0x00)
        self.clear_interrupt()


async def main() -> None:
    detector = CardDetector(
        threshold_mm=DETECTION_THRESHOLD_MM,
        loop_period_ms=INTERMEASUREMENT_MS,
        debounce_s=DEBOUNCE_WINDOW_S,
    )

    print(
        "VL6180X card detector (interrupt + debounce + SHUT pin test):\n"
        f"- Detection threshold: {DETECTION_THRESHOLD_MM} mm\n"
        f"- Debounce window: {DEBOUNCE_WINDOW_S * 1000:.0f} ms\n"
        f"- INT pin: GPIO{INT_PIN} (falling edge)\n"
        f"- SHUT pin: GPIO{SHUT_PIN} (output, active HIGH)\n"
        "After 2 card detections, will test SHUT pin functionality.\n"
        "Press Ctrl+C to stop.\n"
    )

    shut_test_triggered = False

    try:
        with gpiod.request_lines(
            "/dev/gpiochip0",
            consumer="vl6180x-test",
            config={
                INT_PIN: gpiod.LineSettings(
                    edge_detection=Edge.FALLING,
                    bias=Bias.PULL_UP,
                ),
                SHUT_PIN: gpiod.LineSettings(
                    direction=Direction.OUTPUT,
                    output_value=Value.ACTIVE,
                ),
            },
        ) as request:
            print(f"[GPIO] SHUT pin (GPIO{SHUT_PIN}) initialized HIGH (sensor active)")
            detector.set_shut_request(request)

            loop = asyncio.get_event_loop()

            def on_edge_event():
                for event in request.read_edge_events():
                    if event.event_type == event.Type.FALLING_EDGE:
                        detector.handle_interrupt()
                        if detector._last_distance is not None:
                            print(f"[DEBUG] Last distance: {detector._last_distance} mm")

            loop.add_reader(request.fd, on_edge_event)

            try:
                while True:
                    if not shut_test_triggered and detector._card_count >= 2:
                        shut_test_triggered = True
                        print("\n[TRIGGER] 2 cards detected - starting SHUT pin test in 2 seconds...")
                        await asyncio.sleep(2)
                        await detector.test_shut_functionality()

                    await asyncio.sleep(0.5)
            finally:
                loop.remove_reader(request.fd)
    except KeyboardInterrupt:
        pass
    finally:
        detector.shutdown()
        print("\nShutdown complete")


if __name__ == "__main__":
    asyncio.run(main())
