import subprocess
import RPi.GPIO as GPIO
import board
import busio
from lib.motion_handler import MotionHandler
from lib.buzzer import Buzzer
from lib.screen import AutoSizeOLED
from lib.websocket_client import WebSocketClient
import asyncio
from lib.server import ScannerServer

BUZZER_PIN = 26
SERVER_PORT = 8000


async def main():
    GPIO.setmode(GPIO.BCM)
    GPIO.setwarnings(False)

    host_ip = get_ip()

    i2c = busio.I2C(board.SCL, board.SDA)
    i2c_lock = asyncio.Lock()

    ip_text = f"{host_ip}:8000"
    buzzer = Buzzer(BUZZER_PIN)
    screen = AutoSizeOLED(i2c_bus=i2c, i2c_lock=i2c_lock, flip_vertical=True)

    handler = MotionHandler(i2c_bus=i2c, i2c_lock=i2c_lock, buzzer=buzzer, screen=screen)
    await handler.init_gpio()

    ws_client = WebSocketClient(handler=handler, screen=screen, buzzer=buzzer, ip_text=ip_text)
    server = ScannerServer(handler=handler, screen=screen, buzzer=buzzer, ws_client=ws_client)
    ws_client.server = server

    screen.show_text(ip_text)

    # Start both the local web server and the WebSocket client
    asyncio.create_task(server.start())
    asyncio.create_task(ws_client.start())

    try:
        while True:
            await asyncio.sleep(1)
    except KeyboardInterrupt:
        await handler.cleanup()
        await server.cleanup()
        await ws_client.stop()


def get_ip():
    ip = subprocess.check_output(["hostname", "-I"]).decode().split()[0]
    return ip


if __name__ == "__main__":
    asyncio.run(main())
