import adafruit_ssd1306
import board
import busio
from PIL import Image, ImageDraw, ImageFont


class AutoSizeOLED:
    def __init__(self, i2c_addr=0x3C, max_lines=3, flip_vertical=False):
        i2c = busio.I2C(board.SCL, board.SDA)
        self.display = adafruit_ssd1306.SSD1306_I2C(128, 64, i2c, addr=i2c_addr)
        self.font_path = "/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf"
        self.max_lines = max_lines
        self.flip_vertical = flip_vertical

    def _display_image(self, image):
        """Helper method to handle common display logic"""
        self.display.fill(0)
        if self.flip_vertical:
            image = image.transpose(Image.ROTATE_180)
        self.display.image(image)
        self.display.show()

    def show_text(self, text):
        image = Image.new("1", (self.display.width, self.display.height))
        draw = ImageDraw.Draw(image)

        for size in range(40, 8, -2):
            font = ImageFont.truetype(self.font_path, size)

            # Try wrapping at this font size
            wrapped = []
            words = text.split()
            current_line = words[0]

            for word in words[1:]:
                # Try adding the next word
                test_line = current_line + " " + word
                bbox = draw.textbbox((0, 0), test_line, font=font)
                width = bbox[2] - bbox[0]

                if width <= self.display.width:
                    current_line = test_line
                else:
                    wrapped.append(current_line)
                    current_line = word

            wrapped.append(current_line)

            if len(wrapped) > self.max_lines:
                continue

            # Calculate if it fits vertically
            line_height = font.getsize("hg")[1]
            total_height = line_height * len(wrapped)

            if total_height <= self.display.height:
                y = (self.display.height - total_height) // 2
                for line in wrapped:
                    bbox = draw.textbbox((0, 0), line, font=font)
                    text_width = bbox[2] - bbox[0]
                    x = (self.display.width - text_width) // 2
                    draw.text((x, y), line, font=font, fill=255)
                    y += line_height
                break

        self._display_image(image)

    def show_check(self, message=None):
        """Display a large checkmark with optional text below"""
        image = Image.new("1", (self.display.width, self.display.height))
        draw = ImageDraw.Draw(image)

        # Adjust vertical position based on message presence
        base_y = 16 if message else 32
        text_area_height = 20 if message else 0

        # Draw checkmark
        size = 40
        offset = (self.display.width - size) // 2
        draw.line(
            (offset, base_y - size // 4, offset + size // 3, base_y + size // 4),
            fill=255,
            width=3,
        )
        draw.line(
            (offset + size // 3, base_y + size // 4, offset + size, base_y - size // 2),
            fill=255,
            width=3,
        )

        # Add text if provided
        if message:
            self._draw_bottom_text(draw, message, text_area_height)

        self._display_image(image)

    def show_cross(self, message=None):
        """Display a large X with optional text below"""
        image = Image.new("1", (self.display.width, self.display.height))
        draw = ImageDraw.Draw(image)

        # Adjust vertical position based on message presence
        base_y = 16 if message else 32
        text_area_height = 20 if message else 0

        # Draw X
        size = 40
        offset = (self.display.width - size) // 2
        draw.line(
            (offset, base_y - size // 2, offset + size, base_y + size // 2),
            fill=255,
            width=3,
        )
        draw.line(
            (offset + size, base_y - size // 2, offset, base_y + size // 2),
            fill=255,
            width=3,
        )

        # Add text if provided
        if message:
            self._draw_bottom_text(draw, message, text_area_height)

        self._display_image(image)

    def _draw_bottom_text(self, draw, message, text_area_height):
        """Helper to draw text at bottom with automatic sizing"""
        max_font_size = 20
        for size in range(max_font_size, 8, -1):
            font = ImageFont.truetype(self.font_path, size)
            bbox = draw.textbbox((0, 0), message, font=font)
            text_width = bbox[2] - bbox[0]
            text_height = bbox[3] - bbox[1]

            if text_width <= self.display.width and text_height <= text_area_height:
                x = (self.display.width - text_width) // 2
                y = (
                    self.display.height
                    - text_area_height
                    + (text_area_height - text_height) // 2
                )
                draw.text((x, y), message, font=font, fill=255)
                break


# Usage example:
oled = AutoSizeOLED(flip_vertical=True)
oled.show_text("Hello!")  # Will show large
oled.show_check()  # Will show small
oled.show_check("Good")
oled.show_text("This is an even much longer message")  # Will show smaller
oled.show_cross("WAIT")
oled.show_text("again")  # Will show smaller
