# VL6180X Distance Sensor

## Overview

The VL6180X time-of-flight module provides reliable short-range distance measurements for Magic: The Gathering card detection. We run it in continuous-ranging mode while using the sensor's own GPIO1 interrupt so the Raspberry Pi never has to poll over I2C.

## Hardware Snapshot

- **Module**: TOF050C / VL6180X
- **I2C Address**: `0x29`
- **Usable Range**: 5–200 mm
- **Interface**: I2C + GPIO interrupt
- **Interrupt Pin**: Sensor GPIO1 wired to Raspberry Pi GPIO4 (physical pin 7)
- **SHUT Pin**: Hardware enable/reset wired to Raspberry Pi GPIO27 (physical pin 13)

### Wiring Guide

| Sensor Pin | RPi Pin | GPIO | Notes            |
|------------|---------|------|------------------|
| VIN        | Pin 1   | 3.3V | Power            |
| GND        | Pin 6   | GND  | Ground           |
| SDA        | Pin 3   | GPIO2| I2C data         |
| SCL        | Pin 5   | GPIO3| I2C clock        |
| GPIO1/INT  | Pin 7   | GPIO4| Active-low IRQ  |
| SHUT       | Pin 13  | GPIO27| Hardware enable/reset |

## Software Setup

```bash
pip3 install --break-system-packages adafruit-circuitpython-vl6180x
sudo apt install -y i2c-tools

sudo raspi-config nonint do_i2c 0
sudo modprobe i2c-dev
sudo modprobe i2c-bcm2835
```

Verify the bus with `i2cdetect -y 1`; the module should appear at `0x29`.

## SHUT Pin (Hardware Enable/Reset)

The SHUT pin provides hardware-level control of the sensor:

- **HIGH (3.3V)**: Sensor active and operational (normal state)
- **LOW (0V)**: Sensor in hardware shutdown/reset (I2C non-responsive)

### Recovery After SHUT Toggle

When bringing the sensor out of shutdown (SHUT LOW → HIGH), a full re-initialization sequence is required:

1. Pull SHUT HIGH
2. Wait ~100ms for sensor boot
3. Recreate sensor object to trigger fresh-start sequence (clears "fresh out of reset" flag, loads calibration)
4. Reconfigure all sensor registers (interrupt mode, thresholds, continuous ranging)
5. Wait ~500ms for measurements to stabilize

This is necessary because the sensor loses all configuration when SHUT is LOW. The Adafruit library's `__init__` performs critical setup including loading internal calibration and activating the measurement history buffer.

### Use Cases

- **Hardware reset**: Force sensor recovery if it becomes unresponsive
- **Multiple sensors**: Manage I2C address conflicts by sequentially enabling sensors
- **Power management**: Disable sensor when not in use (not currently implemented in production)

## Helper Scripts

- `test_vl6180x.py`: Original proof-of-concept. Uses the Adafruit driver in single-shot mode inside the GPIO callback (software polling).
- `test_vl6180x_v2.py`: Out-of-window interrupt mode test. Configures hardware thresholds so GPIO1 asserts only when distance exits a defined window (dual threshold mode).
- `newer-sensors-tests/test_vl6180x_v2_gpiod.py`: **Current reference implementation**. Uses level-low interrupt mode with software debouncing and includes SHUT pin testing functionality. This matches the production configuration in `lib/card_detector.py`.

We recommend the gpiod version for testing production behavior.

## Production Configuration (Level-Low Mode)

The production code (`lib/card_detector.py`) and current test script (`test_vl6180x_v2_gpiod.py`) use **level-low interrupt mode** with software debouncing:

| Register | Value | Purpose |
|----------|-------|---------|
| `SYSTEM__MODE_GPIO1` | `0x10` | GPIO1 active low, range interrupts enabled |
| `SYSTEM__INTERRUPT_CONFIG_GPIO` | `0x01` | Level-low mode: interrupt fires when distance < threshold |
| `SYSRANGE__THRESH_LOW` | `50` mm | Detection threshold (card present when < 50mm) |
| `SYSRANGE__INTERMEASUREMENT_PERIOD` | `5` ms | Continuous ranging cadence |
| `SYSRANGE__MAX_CONVERGENCE_TIME` | `0x0A` (~10 ms) | Sensor integration window |
| `SYSRANGE__START` | `0x03` | Continuous ranging with immediate start |
| `SYSTEM__INTERRUPT_CLEAR` | `0x07` | Clears latched interrupts after service |

### Runtime Flow

1. Sensor runs continuous distance measurements at ~5 ms intervals with ~10 ms convergence time.
2. When measured distance drops below 50mm, GPIO1 pulls low (active-low interrupt).
3. Raspberry Pi detects falling edge on GPIO4, calls interrupt handler.
4. Handler reads distance, checks interrupt status (expects `0x01` for level-low), and triggers software debouncing.
5. **Software debounce**: Interrupt resets a 200ms timer. Only after 200ms of silence (no new interrupts) is a card detection confirmed.
6. This eliminates false positives from hand jitter or multiple interrupt triggers during single card pass.

No CPU polling occurs except for edge event handling. The debouncing strategy trades slight detection latency for 100% accuracy in noisy physical environments.

## Timing Experiments

| Intermeasurement | Max convergence | Result |
|------------------|-----------------|--------|
| `0` ms | `0x07` (~7 ms) | Works **very well**, detects each card fall; highest responsiveness but noisier readings possible. |
| `5` ms | `0x0A` (~10 ms) | Current baseline. Fast enough for production tests with stable counts. |
| `10` ms | `0x14` (~20 ms) | Observed to miss fast drops; keep as reference only. |

## Usage

```bash
cd /home/academo/scanner-v2-python
python3 newer-sensors-tests/test_vl6180x_v2_gpiod.py
```

Key constants live at the top of the script:

```python
INT_PIN = 4                    # Interrupt GPIO pin
SHUT_PIN = 27                  # Hardware enable/reset pin
DETECTION_THRESHOLD_MM = 50    # Trigger interrupt when distance < 50mm
INTERMEASUREMENT_MS = 5        # Measurement interval
DEBOUNCE_WINDOW_S = 0.2        # Software debounce period
```

The test script will automatically trigger a SHUT pin test after detecting 2 cards, demonstrating the hardware reset functionality. Adjust these values to match your mechanical setup and lighting conditions.

## Notes & Next Steps

- Always clean up GPIO via Ctrl+C; the script's signal handlers stop continuous ranging and release GPIO4 automatically.
- If future tuning requires more or less sensitivity, adjust `DETECTION_THRESHOLD_MM` (currently 50mm) or `DEBOUNCE_WINDOW_S` (currently 200ms).
- Shorter intermeasurement periods increase responsiveness but may introduce noise. Current 5ms baseline works well for production.
- The SHUT pin can be used for hardware recovery if the sensor becomes unresponsive, though this is not currently implemented in production scanner code.
- Keep older test scripts around for reference, but treat `newer-sensors-tests/test_vl6180x_v2_gpiod.py` as the production-targeted reference implementation.
